<?php

namespace App\Http\Controllers\Configuracion;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use App\Models\User;
use Spatie\Permission\Models\Permission;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;
use DataTables;

class UsuariosController extends Controller {

    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            if (!Auth::user()->hasPermissionTo('Usuarios')) {
                return response()->view('errors.403', [
                    'modulo' => 'Usuarios'
                ], 403);
            }

            return $next($request);
        });
    }

    public function index()
    {
        if(Auth::user()->hasRole('Supervisor')){
            $roles = DB::table("roles")->whereIn("name", ["Supervisor", "Operador", "Gerencia"])->get();
        }else{
            $roles = DB::table("roles")->get();
        }

        $empresas = DB::table("con_empresa")->where("Hab", 1)->get();
        return view('configuracion.usuarios.index')
            ->with('empresas', $empresas)
            ->with('roles', $roles);
    }

    public function get(){
        $usuarios = DB::table('con_usuarios');
        $usuarios = $usuarios->where("Hab", 1);
        if(!Auth::user()->hasRole('Administrador')){
            $usuarios = $usuarios->where("idEmpresa", Auth::user()->idEmpresa);
        }
        $usuarios = $usuarios->orderBy("id", "ASC");
        $usuarios = $usuarios->get(); 
        return Datatables::of($usuarios)
            ->addIndexColumn()
            ->addColumn('Accion', function($row){
                return "";
            })
            ->rawColumns(['action'])
            ->make(true);
    }

    public function edit($id){
        $usuario = DB::table('con_usuarios');
        $usuario = $usuario->where("id", $id);
        $usuario = $usuario->first();
        return response()->json([
            'success' => true,
            'data' => $usuario,
        ]);
    }

    public function store(Request $request){
        
        $rules = [
            'nombre'      => 'required',
            'apellido'    => 'required',
            'usuario'     => 'required',
            'tipoUsuario' => 'required',
        ];

        if (!$request->id) {
            $rules['email'] = [
                'required',
                'email',
                Rule::unique('con_usuarios')->where(fn ($q) => $q->where('idEmpresa', $request->idEmpresa)),
            ];
            $rules['clave'] = 'required|min:8';
            $rules['clave2'] = 'required|same:clave';
        }else{

            $rules['email'] = [
                'required',
                'email',
                Rule::unique('con_usuarios')->ignore($request->id, 'id')->where(fn ($q) => $q->where('idEmpresa', $request->idEmpresa)),
            ];

            if ($request->filled(['clave', 'clave2'])) {
                $rules['clave'] = 'min:8';
                $rules['clave2'] = 'same:clave';
            }

        }

        $messages = [
            'required' => 'El campo :attribute es obligatorio.',
            'min' => 'El campo :attribute debe tener al menos :min caracteres.',
            'same' => 'Las claves no coinciden.',
            'email' => 'El email no es válido.',
            'email.unique'  => 'Ya existe un usuario con ese correo en esta empresa.',
        ];

        $attributes = [
            'idEmpresa'   => 'empresa',
            'nombre'      => 'nombre',
            'apellido'    => 'apellido',
            'usuario'     => 'usuario',
            'email'       => 'email',
            'tipoUsuario' => 'tipo usuario',
            'clave'      => 'clave',
            'clave2'      => 'repetir clave',
        ];

        if(Auth::user()->hasRole('Administrador')){
            $rules['idEmpresa'] = 'required';
            $attributes['idEmpresa'] = 'empresa';
            $idEmpresa = $request->idEmpresa;
        }else{
            $idEmpresa = Auth::user()->idEmpresa;
        }

        $validator = Validator::make($request->all(), $rules, $messages);
        $validator->setAttributeNames($attributes);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'error' => $validator->errors()->all()
            ]);
        }
        
        if(!$request->id){
            $insertId = DB::table('con_usuarios')->insertGetId([
                'idEmpresa' => $idEmpresa,
                'nombre' => $request->nombre,
                'apellido' => $request->apellido,
                'email' => $request->email,
                'usuario' => $request->usuario,
                'tipoUsuario' => $request->tipoUsuario,
                'departamento' => $request->departamento,
                'telefonos' => $request->telefonos,
                'clave' => bcrypt($request->clave),
                'hab' => $request->hab
            ]);
            if($insertId){
                $nuevoUsuario = User::find($insertId);
                if (in_array($request->tipoUsuario, ['Administrador', 'Supervisor', 'Operador'])) {
                    $nuevoUsuario->assignRole($request->tipoUsuario);
                }

                $response = [
                    "success" => true,
                    "message" => "Registro #{$insertId} creado con exito"
                ];
            }
        }else{

            $datosUpdate = [
                'idEmpresa' => $idEmpresa,
                'nombre' => $request->nombre,
                'apellido' => $request->apellido,
                'email' => $request->email,
                'usuario' => $request->usuario,
                'tipoUsuario' => $request->tipoUsuario,
                'departamento' => $request->departamento,
                'telefonos' => $request->telefonos,
                'hab' => $request->hab
            ];

            if ($request->filled(['clave', 'clave2'])) {
                $datosUpdate['clave'] = bcrypt($request->clave);
            }

            DB::table('con_usuarios')->where("id", $request->id)->update($datosUpdate);    
            
            $usuario = User::find($request->id);

            if ($usuario && in_array($request->tipoUsuario, ['Administrador', 'Supervisor', 'Operador'])) {
                $usuario->syncRoles([$request->tipoUsuario]);
            }

            $response = [
                "success" => true,
                "message" => "Registro #{$request->id} modificado con exito"
            ];
        }
        return response()->json($response);

    }

    public function destroy($id){
        $Empresa = DB::table("con_usuarios")->where("id", $id)->update([
            'Hab' => 0
        ]);    
        $response = [
            'success' => true,
            'message' => 'Usuario inhabilitado con exito'
        ];
        return response()->json($response);
    }
}
